#!/usr/bin/perl -w
# The summarization module of the sRNAnalyzer pipeline
# ARGV[0] - a database config file
# ARGV[1] - a sample order file
# Options: --project NAME - gives the project name to be used in output filenames

use strict;
use warnings;
use Getopt::Long qw(GetOptions);
use FindBin qw($Bin); # $Bin is now the directory where the script is

# read the project name from the command line
my $project_name = "all_sample";
my $preprocess_only = '';
my $align_only = '';
my $sum_miRNA;
GetOptions("project=s" => \$project_name, "preprocess-only" => \$preprocess_only, "align-only" => \$align_only,
           "miRNA" => \$sum_miRNA);

sub usage {
    print "Incorrect number of arguments\n";
    print "Usage: $0 <db-config-file> <sample-order-file> --project <project-name>\n\n";
    print "Arguments:\n";
    print "db-config-file    - path of the database configuration file (required) \n";
    print "sample-order-file - file specifying order of the samples (optional). If blank, alphabetical order is used\n";
    print "project-name      - prefix to be used for output files (optional). Default prefix is all_samples\n";
    exit 1;
}

# get the DB config file
my $db_config_file = $ARGV[0];

if (not defined $db_config_file) {
    # user didn't give a db config file, so show the usage message
    usage();
    exit 1;
}

# set the sample order if it is given
my $sampleOrder = $ARGV[1];
if (not $sampleOrder) {
    $sampleOrder = "";
}

my @args = ($sampleOrder, "--project", $project_name);

# summarize trimming reports
if (not $align_only) {
    system("$Bin/selfDev/sumTrim.pl", @args);
    system("$Bin/selfDev/sumReadN.pl", @args);
}

if (not $preprocess_only) {
    # summarize read distribution and match rate
    system("$Bin/selfDev/sumDist.pl", @args);
    system("$Bin/selfDev/sumRate.pl", @args);
    
    # summarize features and profiles
    system("$Bin/selfDev/sumFeature.pl", @args);
    system("$Bin/selfDev/sumProfile.pl", @args);
    
    # add descriptions to the feature files
    system("$Bin/selfDev/desFeature.pl", ("$project_name.feature", $db_config_file));
    system("$Bin/selfDev/desProfile.pl", ("$project_name.profile", $db_config_file));
}

if ($sum_miRNA) {
    # summarize miRNA, including matchID, and SNPs
    system("$Bin/selfDev/sumMatchID_miR.pl", ("$project_name.feature", "--FW", "--mis_0", "--mirID", "--weightMismatch"));
    system("$Bin/selfDev/sumMatchID_miR.pl", ("$project_name.feature", "--FW", "--mis_1", "--mirID", "--weightMismatch"));
    system("$Bin/selfDev/sumMatchID_miR.pl", ("$project_name.feature", "--FW", "--mis_2", "--mirID", "--weightMismatch"));
    system("$Bin/selfDev/sumDist_miR_v1.7.pl", ($db_config_file, $sampleOrder, "--project", $project_name));
    chdir("Dist_miR_mis_1+");
    system("$Bin/selfDev/verifyDist_miR.pl", ($db_config_file, "--project", $project_name));
}
