#!/usr/bin/perl -w

#===============================================================================
# Single-end (SE) read sequence preprocessing for , including:
# for FASTA files, 3' adapters has already been trimmed and may also be collased
# 1) use Cutadapt for  5' adatper trmming
# 2) use Cutadapt for empty read (only adapter but no insert sequence) discarding
# 3) use Prinseq for low-quality read filtering
# 4) use FASTX_collapser for identical read collapsing
# 5) use Cutadapt for stop oligo filtering (optional)
# mainly for NEB Next small RNA kit and Illumina TruSeq small RNA kit. 
# 	NEB Next Small RNA Sample Prep Kit
# 		3’ Adapter: AGATCGGAAGAGCACACGTCT
#		e.g., AGATCGGAAGAGCACACGTCT GAACTCCAGTCAC XXXXXX ATCTCGTATGCC GTCTTCTGCTTG
#										barcode - XXXXXX
# 		5’ Adapter: GTTCAGAGTTCTACAGTCCGACGATC
# 	Illumina TruSeq Small RNA Sample Prep Kit
# 		3’ Adapter: TGGAATTCTCGGGTGCCAAG
#		e.g., TGGAATTCTCGGGTGCCAAG GAACTCCAGTCAC XXXXXX ATCTCGTATGCC GTCTTCTGCTTG
#									   barcode - XXXXXX
# 		5’ Adapter: GTTCAGAGTTCTACAGTCCGACGATC
#===============================================================================

use strict;
use warnings;

my $option_01 = $ARGV[0];
if (scalar(@ARGV) == 0) {
	$option_01 = "null";
}

my $dataDIR = "./";

opendir my $dir, $dataDIR or die "Cannot open directory: $dataDIR";
my @fileList = readdir $dir;
closedir $dir;

my $suffix = "\.fa";
my @readFileList = grep(/$suffix$/, @fileList);
@readFileList = sort @readFileList;

my $num = scalar(@readFileList);
if ( $num == 0 ) {
	print "\nThere is NO FASTQ file ($suffix) in this folder\n";
	exit;
} elsif ( $num == 1 ) {
	print "\nThere is ".$num." FASTQ file ($suffix) in this folder\n";
} else {
	print "\nThere are ".$num." FASTQ file ($suffix) in this folder\n";
}

# parameters for cutadapt
my $adapter_3 = "AGATCGGAAGAGCACACGTCT";
my $adapter_5 = "GTTCAGAGTTCTACAGTCCGACGATC";
my $emptyIn_3 = "AGATCGGAAGAGCACACG";
my $emptyIn_5 = "AGAGTTCTACAGTCCGA";
my $stop_oligo_A = "CGTTCCCGTGG\$";
my $stop_oligo_B = "CCACGTTCCCG";
my $err_3 = 0.2;
my $err_5 = 0.125;
my $err_BC = 0.0625;
my $overlap_3 = 5;
my $overlap_5 = 8;
my $overlap_BC = 12;
my $overlap_STP = 11;
my $min_length = 1;

# parameters for prinseq
my $lc_method = "entropy";
my $lc_threshold = 50;
my $min_len = 15;
my $trim_tail_right = 5;
my $line_width = 60;

# read sample barcode file
my %sampleBarcode = ();
open(In, "<", "sampleBarcode") or die "\n   can't find sampleBarcode file!!!\n";
while (<In>) {
	chomp;
	my @array = split(/\t/, $_);
	if ( scalar(@array) == 2 ) {
		$sampleBarcode{$array[0]} = $array[1];
	}
}
close(In);

my $i = 0;
foreach my $readFile (@readFileList) {
	$i++;
	my $prefix = $readFile;
	$prefix =~ s/$suffix$//;
	
	my $barcode;
	if ( exists $sampleBarcode{$prefix} ) {
		$barcode = $sampleBarcode{$prefix};
	} else {
		print "\n   there's no barcode for sample: $prefix!!!\n";
		next;
	}
	
	print "\n====== Preprocess #".$i." read file: $prefix ======\n";
	preprocess($prefix, $suffix, $barcode);
}

#================================ sub functions ================================
# preprocess
sub preprocess {
	my ($prefix, $suffix, $barcode) = @_;
	
	my $readFile = $prefix.$suffix;
	
	my $barcodeArea_1 = $barcode."ATCTCGTATGCCG";
	my $barcodeArea_2 = "CAGTCAC".$barcode."ATCTCGT";
	my $barcodeArea_3 = "GAACTCCAGTCAC".$barcode;

	my $expandedReadFile = $prefix."\_Expanded.fasta";
	my $trimedReadFile = $prefix."\_Cutadapt.fasta";
	my $trimReport = $prefix."\_Cutadapt.report";
	
	my $filteredReadFile = $prefix."\_Prinseq";
	my $filterReport = $prefix."\_Cutadapt_Prinseq.report";
	
	my $collapsedReadFile = $prefix."\_Collapsed.fa";
	my $processedReadFile = $prefix."\_Processed.fa";
	my $stpReadFile0 = $prefix."\_stp_mis_0.fa";
	my $stpReadFile1 = $prefix."\_stp_mis_1.fa";
	my $stpReadFile2 = $prefix."\_stp_mis_2.fa";
	my $stpReport = $prefix."\_stp.report"; 
	
	# expand collased FASTA file (sequence head with read number 'NO-N', one sequence N reads) to general FASTA file (sequence head without read number, one sequence one read)
	expandFASTA($readFile, $expandedReadFile);
	
	# use Cutadapt for 5' adatper trmming
	system("cutadapt -g $adapter_5 -n 1 -e $err_5 -O $overlap_5 -m $min_length --match-read-wildcards -o tmp_Cutadapt.fasta $expandedReadFile > $trimReport");
	
	# use Cutadapt for empty read (only adapter but no insert sequence) discarding
	system("cutadapt -b $barcodeArea_1 -b $barcodeArea_2 -b $barcodeArea_3 -b $emptyIn_3 -b $emptyIn_5 -n 1 -e $err_BC -O $overlap_BC -m $min_length --match-read-wildcards --discard-trimmed -o $trimedReadFile tmp_Cutadapt.fasta >> $trimReport");
	
	system("rm $expandedReadFile tmp_Cutadapt.fasta");
	print "   --- Adatper trmming for $prefix completed\n";
	
	# use Prinseq for low-quality read filtering
	system("prinseq-lite.pl -fasta $trimedReadFile -out_format 1 -out_good $filteredReadFile -out_bad null -lc_method $lc_method -lc_threshold $lc_threshold -min_len $min_len -line_width $line_width -trim_tail_right $trim_tail_right 2> $filterReport");

	system("rm $trimedReadFile");
	print "   --- Low-quality read filtering for $prefix completed\n";

	# use FASTX_collapser for identical read collapsing
	system("fastx_collapser -i $filteredReadFile\.fasta -o $collapsedReadFile");
	
	system("rm $filteredReadFile\.fasta");
	print "   --- Identical read collapsing for $prefix completed\n";
	
	# use Cutadapt for stop oligo filtering
	if ($option_01 eq "--stp") {
		system("cutadapt -b $stop_oligo_A -b $stop_oligo_B -n 1 -e 0.000 -O $overlap_STP --no-trim --untrimmed-output stp0_Processed.fa -o $stpReadFile0 $collapsedReadFile >> $stpReport");	# maxMismatch = 0
		system("cutadapt -b $stop_oligo_A -n 1 -e 0.091 -O $overlap_STP --no-trim --untrimmed-output stp1_Processed.fa -o $stpReadFile1 stp0_Processed.fa >> $stpReport");						# maxMismatch = 1
		system("cutadapt -b $stop_oligo_A -n 1 -e 0.182 -O $overlap_STP --no-trim --untrimmed-output $processedReadFile -o $stpReadFile2 stp1_Processed.fa >> $stpReport");						# maxMismatch = 2
		
		system("rm $collapsedReadFile stp0_Processed.fa stp1_Processed.fa");
		print "   --- Stop oligo trimming for $prefix completed\n";
	} else {
		system("mv $collapsedReadFile $processedReadFile");
	}
}

# expand collased FASTA file (sequence head with read number 'NO-N', one sequence N reads) to general FASTA file (sequence head without read number, one sequence one read)
sub expandFASTA {
	my ($readFile, $expandedReadFile) = @_;

	# read collapsed FASTA file and write expanded FASTA file
	open(In, "<", "$readFile") or die "\n   can't find $readFile!!!\n";
	open(Out, ">", "$expandedReadFile");
	my $readID = "1-1";
	my $readNum = 1;
	# sum readN for all reads (collapsed or general)
	while (<In>) {
		chomp;
		if ( /^>/ ) {	# sequence head
			$readID = $_;
			my @head = split(/\-/, $readID);
			if ( (scalar(@head) == 2) and ($head[1]=~ /^\d+$/) ) {	# if it's collapsed FASTA format
				$readID = $head[0];
				$readNum = $head[1];
			} else { 												# if it's NOT a collapsed FASTA format
				$readNum = 1;
			}
		} else {		# sequence
			my $readSeq = $_;
			for my $i (1..$readNum) {
				print Out $readID."\_".$i."\n";
				print Out $readSeq."\n";
			} 
		}
	}
	close(In);
	close(Out);
}