#!/usr/bin/perl -w

#===============================================================================
# Paired-end read sequence preprocessing, including:
# 1) use Cutadapt for 3' adapter trmming and 5' adatper trmming
# 2) use Cutadapt for empty read (only adapter but no insert sequence) discarding
# 3) use Prinseq for low-quality read filtering
# 4) use FASTX_collapser for identical read collapsing
# mainly for NEB Paired-end (PE) RNA Adapters
# 	NEB Paired-end (PE) RNA Adapters
# 		3’ Adapter: GATCGGAAGAGCACACGTCT GAACTCCAGTC
# 		5’ Adapter: GATCGGAAGAGC GTCGTGTAGGGAAAGAGTGT
# 	Illumina PCR Primers
# 		P7_PCR_primer (ligated on the left of 5’ Adapter):	ATCTCGTATGCCGTCTTCTGCTTG
# 		P5_PCR_primer (ligated on the right of 3’ Adapter):	AGATCTCGGTGGTCGCCGTATCATT
#===============================================================================

use strict;
use warnings;

my $option_01 = $ARGV[0];
if ( !$ARGV[0] ) {
	$option_01 = "-z";
}

my $dataDIR = "./";

opendir my $dir, $dataDIR or die "Cannot open directory: $dataDIR";
my @fileList = readdir $dir;
closedir $dir;

my $suffix = "\.fastq";
if ( $option_01 eq "-z" ) {
	$suffix = "\.fastq.gz";
}
my @readFileList = grep(/$suffix$/, @fileList);
@readFileList = sort @readFileList;

my $num = scalar(@readFileList);
if ( $num == 0 ) {
	print "\nThere is NO FASTQ file ($suffix) in this folder\n";
	exit;
} elsif ( $num == 1 ) {
	print "\nThere is ".$num." FASTQ file ($suffix) in this folder\n";
} else {
	print "\nThere are ".$num." FASTQ file ($suffix) in this folder\n";
}

# parameters for cutadapt
my $adapter_PE_3 = "GATCGGAAGAGCACACGTCT";
my $adapter_PE_5 = "GTCGTGTAGGGAAAGAGTGT";
my $P7_PCR_primer = "ATCTCGTATGCCGTCTTCTGCTTG";
my $P5_PCR_primer = "AGATCTCGGTGGTCGCCGTATCATT";

my $err_3 = 0.2;
my $err_5 = 0.125;
my $err_BC = 0.0625;
my $overlap_3 = 5;
my $overlap_5 = 8;
my $overlap_BC = 12;
my $min_length = 1;


# parameters for prinseq
my $lc_method = "entropy";
my $lc_threshold = 50;
my $min_len = 15;
my $trim_tail_right = 5;
my $line_width = 80;

my $i = 0;
foreach my $readFile (@readFileList) {
	$i++;
	my $prefix = $readFile;
	$prefix =~ s/$suffix$//;

	print "\n====== Preprocess #".$i." read file: $prefix ======\n";
	preprocess($prefix, $suffix);
}

#================================ sub functions ================================
# preprocess
sub preprocess {
	my ($prefix, $suffix) = @_;
	
	my $readFile = $prefix.$suffix;
	
	my $trimedReadFile = $prefix."\_Cutadapt.fastq";
	my $trimReport = $prefix."\_Cutadapt.report";
	
	my $filteredReadFile = $prefix."\_Prinseq.fastq";
	my $filterReport = $prefix."\_Cutadapt_Prinseq.report";
	
	my $collapsedReadFile = $prefix."\_Processed.fa";
	
	# use Cutadapt for 3' adapter trmming and 5' adatper trmming
	system("cutadapt -a $adapter_PE_3 -e $err_3 -O $overlap_3 -m $min_length --match-read-wildcards -o tmp_Cutadapt3.fastq $readFile > $trimReport");
	system("cutadapt -g $adapter_PE_5 -e $err_5 -O $overlap_5 -m $min_length --match-read-wildcards -o tmp_Cutadapt.fastq tmp_Cutadapt3.fastq >> $trimReport");
	
	# use Cutadapt for empty read (only adapter but no insert sequence) discarding
	system("cutadapt -b $P7_PCR_primer -b $P5_PCR_primer -e $err_BC -O $overlap_BC -m $min_length --match-read-wildcards --discard-trimmed -o $trimedReadFile tmp_Cutadapt.fastq >> $trimReport");

	system("rm tmp_Cutadapt3.fastq tmp_Cutadapt.fastq");
	print "   --- Adatper trmming for $prefix completed\n";
	
	# use Prinseq for low-quality read filtering
	system("prinseq-lite.pl -fastq $trimedReadFile -out_format 1 -out_good $filteredReadFile -out_bad null -lc_method $lc_method -lc_threshold $lc_threshold -min_len $min_len -line_width $line_width -trim_tail_right $trim_tail_right 2> $filterReport");

	system("rm $trimedReadFile");
	print "   --- Low-quality read filtering for $prefix completed\n";

	# use FASTX_collapser for identical read collapsing
	system("fastx_collapser -i $filteredReadFile\.fasta -o $collapsedReadFile");
	
	system("rm $filteredReadFile\.fasta");
	print "   --- Identical read collapsing for $prefix completed\n";
}
