#!/usr/bin/perl -w

#===============================================================================
# Paired-end read sequence preprocessing, including:
# 1) use Cutadapt for 3' adapter trmming and 5' adatper trmming
# 2) use Cutadapt for empty read (only adapter but no insert sequence) discarding
# 3) use Prinseq for low-quality read filtering
# 4) use FASTX_collapser for identical read collapsing
# mainly for Rubicon Genomics Universal Primers (Pre-amplification), Illumina Paired-end (PE) DNA Adapters, Illumina TruSeq universal DNA/RNA kits. 
#	Rubicon Genomics Universal Primers for pre-amplification
#		FW: CCAAACACACCCAACACACCAC
#		RC: GTGGTGTGTTGGGTGTGTTTGG
# 	Illumina Paired-end (PE) DNA Adapters
# 		3’ Adapter: AGATCGGAAGAGCGGTTCAGCAGGAATGCCGAG
# 		5’ Adapter: AGATCGGAAGAGCGTCGTGTAGGGAAAGAGTGT
# 	Illumina TruSeq universal DNA/RNA kits
# 		3’ Adapter: AGATCGGAAGAGCACACGTCTGAACTCCAGTCAC
#			e.g., AGATCGGAAGAGCACACGTCTGAACTCCAGTCAC XXXXXX ATCTCGTATGCCGTCTTCTGCTTG
#											barcode: XXXXXX
# 		5’ Adapter: GTTCAGAGTTCTACAGTCCGACGATC
# 	Illumina PCR Primers
# 		P7_PCR_primer (ligated on the left of 5’ Adapter):	ATCTCGTATGCCGTCTTCTGCTTG
# 		P5_PCR_primer (ligated on the right of 3’ Adapter):	AGATCTCGGTGGTCGCCGTATCATT
#===============================================================================

use strict;
use warnings;

my $option_01 = $ARGV[0];
if ( !$ARGV[0] ) {
	$option_01 = "-z";
}

my $dataDIR = "./";

opendir my $dir, $dataDIR or die "Cannot open directory: $dataDIR";
my @fileList = readdir $dir;
closedir $dir;

my $suffix = "\.fastq";
if ( $option_01 eq "-z" ) {
	$suffix = "\.fastq.gz";
}
my @readFileList = grep(/$suffix$/, @fileList);
@readFileList = sort @readFileList;

my $num = scalar(@readFileList);
if ( $num == 0 ) {
	print "\nThere is NO FASTQ file ($suffix) in this folder\n";
	exit;
} elsif ( $num == 1 ) {
	print "\nThere is ".$num." FASTQ file ($suffix) in this folder\n";
} else {
	print "\nThere are ".$num." FASTQ file ($suffix) in this folder\n";
}

# parameters for cutadapt
my $primer_FW = "CCAAACACACCCAACACACCAC";
my $primer_RC = "GTGGTGTGTTGGGTGTGTTTGG";
my $adapter_PE_3 = "AGATCGGAAGAGCGGTTCAGCAGGAATGCCGAG";
my $adapter_PE_5 = "AGATCGGAAGAGCGTCGTGTAGGGAAAGAGTGT";
my $adapter_Uni_3a = "AGATCGGAAGAGCACACGTCT";
my $adapter_Uni_3b = "GATCGGAAGAGCACACGTCT";
my $adapter_Uni_3c = "AAGAGCACACGTCTGAACTCC";
my $adapter_Uni_3d = "CACACGTCTGAACTCCAGTCAC";
my $adapter_Uni_3e = "GAACTCCAGTCAC";
my $adapter_Uni_5 = "GTTCAGAGTTCTACAGTCCGACGATC";
my $adapter_TruSeq_3a = "TGGAATTCTCGGGTGCCAAG";
my $adapter_TruSeq_3b = "TCTCGGGTGCCAAGGAACTCC";
my $adapter_TruSeq_3c = "GGTGCCAAGGAACTCCAGTCAC";
my $P7_PCR_primer = "ATCTCGTATGCCGTCTTCTGCTTG";
my $P5_PCR_primer = "AGATCTCGGTGGTCGCCGTATCATT";

my $err_3 = 0.2;
my $err_5 = 0.2;
my $err_BC = 0.1;
my $overlap_3 = 5;
my $overlap_5 = 5;
my $overlap_BC = 10;
my $min_length = 1;


# parameters for prinseq
my $lc_method = "entropy";
my $lc_threshold = 50;
my $min_len = 15;
my $trim_tail_right = 5;
my $line_width = 55;

# read sample barcode file
my %sampleBarcode = ();
open(In, "<", "sampleBarcode_S") or die "\n   can't find sampleBarcode file!!!\n";
while (<In>) {
	chomp;
	my @array = split(/\t/, $_);
	if ( scalar(@array) == 2 ) {
		$sampleBarcode{$array[0]} = $array[1];
	}
}
close(In);

my $i = 0;
foreach my $readFile (@readFileList) {
	$i++;
	my $prefix = $readFile;
	$prefix =~ s/$suffix$//;
	
	my $barcode;
	if ( exists $sampleBarcode{$prefix} ) {
		$barcode = $sampleBarcode{$prefix};
	} else {
		print "\n   there's no barcode for sample: $prefix!!!\n";
		next;
	}
	
	print "\n====== Preprocess #".$i." read file: $prefix ======\n";
	preprocess($prefix, $suffix, $barcode);
}

#================================ sub functions ================================
# preprocess
sub preprocess {
	my ($prefix, $suffix, $barcode) = @_;
	
	my $readFile = $prefix.$suffix;
	
	my $barcodeArea_1 = $barcode."ATCTCGTATGCCG";
	my $barcodeArea_2 = "CAGTCAC".$barcode."ATCTCGT";
	my $barcodeArea_3 = "GAACTCCAGTCAC".$barcode;
	
	my $trimedReadFile = $prefix."\_Cutadapt.fastq";
	my $trimReport = $prefix."\_Cutadapt.report";
	
	my $filteredReadFile = $prefix."\_Prinseq.fastq";
	my $filterReport = $prefix."\_Cutadapt_Prinseq.report";
	
	my $collapsedReadFile = $prefix."\_Processed.fa";
	
	# use Cutadapt for 3' adapter trmming and 5' adatper trmming
	system("cutadapt -a $primer_FW -a $adapter_PE_3 -a $adapter_Uni_3a -a $adapter_Uni_3b -a $adapter_Uni_3c -a $adapter_Uni_3d -a $adapter_Uni_3e -a $adapter_TruSeq_3a -a $adapter_TruSeq_3b -a $adapter_TruSeq_3c -n 10 -e $err_3 -O $overlap_3 -m $min_length --match-read-wildcards -o tmp_Cutadapt3.fastq $readFile > $trimReport");
	system("cutadapt -g $primer_RC -g $adapter_PE_5 -g $adapter_Uni_5 -n 3 -e $err_5 -O $overlap_5 -m $min_length --match-read-wildcards -o tmp_Cutadapt.fastq tmp_Cutadapt3.fastq >> $trimReport");
	
	# use Cutadapt for empty read (only adapter but no insert sequence) discarding
	system("cutadapt -b $primer_FW -b $primer_RC -b $P7_PCR_primer -b $P5_PCR_primer -b $barcodeArea_1 -b $barcodeArea_2 -b $barcodeArea_3 -n 7 -e $err_BC -O $overlap_BC -m $min_length --match-read-wildcards --discard-trimmed -o $trimedReadFile tmp_Cutadapt.fastq >> $trimReport");

	system("rm tmp_Cutadapt3.fastq tmp_Cutadapt.fastq");
	print "   --- Adatper trmming for $prefix completed\n";
	
	# use Prinseq for low-quality read filtering
	system("prinseq-lite.pl -fastq $trimedReadFile -out_format 1 -out_good $filteredReadFile -out_bad null -lc_method $lc_method -lc_threshold $lc_threshold -min_len $min_len -line_width $line_width -trim_tail_right $trim_tail_right 2> $filterReport");

	system("rm $trimedReadFile");
	print "   --- Low-quality read filtering for $prefix completed\n";

	# use FASTX_collapser for identical read collapsing
	system("fastx_collapser -i $filteredReadFile\.fasta -o $collapsedReadFile");
	
	system("rm $filteredReadFile\.fasta");
	print "   --- Identical read collapsing for $prefix completed\n";
}
