#!/usr/bin/perl -w
# 1st argument - the data folder
# 2nd argument - the path to the pipeline config file
# 3rd argument - the path to the DB name and path config file 

use strict;
use warnings;
use FindBin qw($Bin);
use lib "$Bin/Tools"; # add sRNAnalyzer/Tools to the loading module loading path
use YAML::Tiny; # load the YAML::Tiny module
use File::Spec;

# if wrong number of arguments
# show usage message
if (scalar(@ARGV) != 3) {
	print "Incorrect number of arguments\n";
	print "Usage: $0 DATA_FOLDER CONFIG_FILE DB_CONFIG_FILE\n\n";
	print "Where DATA_FOLDER is a folder containing _Processed.fa files, \n";
	print "CONFIG_FILE is the configuration file for the pipeline, and\n";
	print "DB_CONFIG_FILE is the file containing the names and paths of databases.\n\n";
	exit;
}

my $dataDIR = $ARGV[0];
my $configFilePath = $ARGV[1];
my $dbFilePath = $ARGV[2];

# use YAML tiny to read the assignment type, single or multipel
my $yaml = YAML::Tiny->read($configFilePath);
my $config = $yaml->[0];
my $assignmentType = $config->{"alignment"}->{"type"};
my $num_cores = $config->{"alignment"}->{"cores"};

if (not $num_cores) {
	$num_cores = 15;
}

my $source_file;
if ($assignmentType) {
	if ($assignmentType eq "multiple") {
		# use multiple assignment (weighted)
		$source_file = "matchRNA_v1.6w_miR.pl";
	}
	elsif ($assignmentType eq "single") {
		# use single assignment
		$source_file = "matchRNA_v1.6.5.pl";
	}
	else {
		print "Invalid assignment type provided: $assignmentType\n";
		print qq(Choose either "single" or "multiple" \n);
	}
}
else {
	# no assignment type provided in the config file
	# use single assignment by default
	$source_file = "matchRNA_v1.6.5.pl";
}

# open and read files from the directory specified
opendir my $dir, $dataDIR or die "Cannot open directory: $dataDIR";
my @fileList = readdir $dir;
closedir $dir;

# find files ending with _Processed.fa
my @readFileList = grep(/\_Processed.fa$/, @fileList);
@readFileList = sort @readFileList;

print join("\t", @readFileList);
my $num = scalar(@readFileList);

if ( $num == 0 ) {
	print "\n\nThere is NO preprocessed FASTA file (_Processed.fa) in this folder\n";
	exit;
} elsif ( $num == 1 ) {
	print "\n\nThere is ".$num." preprocessed FASTA file (_Processed.fa) in this folder\n";
} else {
	print "\n\nThere are ".$num." preprocessed FASTA files (_Processed.fa) in this folder\n";
	}

# get the directory that this source file is in
my $i = 0;
my $maxReadLength = 86;
foreach my $readFile (@readFileList) {
	$i++;
	print "\n====== Begin to match #".$i." read file ======";
	my $perl_file = "$Bin/selfDev/$source_file";
	my $fa_file = File::Spec->catfile($dataDIR, $readFile);
	my @args = ($maxReadLength, $fa_file, $configFilePath, $dbFilePath, "--cores", $num_cores);
	system($perl_file, @args);
}

#system("mkdir PisaR");
#system("mv *.profile ./PisaR/");
#system("mv *.feature ./PisaR/");
#system("mv *.dist ./PisaR/");
#system("mv *_unMatch.fa ./PisaR/");
